<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\CurlRequest;
use App\Lib\MultiVideoUploader;
use App\Models\Category;
use App\Models\Episode;
use App\Models\Item;
use App\Models\Reel;
use App\Models\SubCategory;
use App\Models\Subscriber;
use App\Models\Subtitle;
use App\Models\User;
use App\Models\Video;
use App\Models\VideoReport;
use App\Rules\FileTypeValidate;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class ItemController extends Controller {

    public function items() {
        return $this->renderItemsView("Play Video Items");
    }

    public function singleItems() {
        return $this->renderItemsView("Single Video Items", 'singleItems');
    }

    public function episodeItems() {
        return $this->renderItemsView("Episode Video Items", 'episodeItems');
    }

    public function trailerItems() {
        return $this->renderItemsView("Trailer Video Items", 'trailerItems');
    }

    public function rentItems() {
        return $this->renderItemsView("Rent Video Items", 'rentItems');
    }

    private function renderItemsView($pageTitle, $scope = null) {
        $items = $this->itemsData($scope);
        return view('admin.item.index', compact('pageTitle', 'items'));
    }

    private function itemsData($scope = null) {
        $query = Item::with('category', 'sub_category', 'video');
        if ($scope) {
            $query = $query->$scope();
        }
        return $query->searchable(['title', 'category:name'])->filter(['version', 'item_type', 'featured', 'category_id'])->orderBy('id', 'desc')->paginate(getPaginate());
    }

    private function itemValidation($request, $type) {
        $validation = $type == 'create' ? 'required' : 'nullable';
        $versions   = implode(',', [Status::FREE_VERSION, Status::PAID_VERSION, Status::RENT_VERSION]);
        $request->validate([
            'title'           => 'required|string|max:255',
            'category'        => 'required|integer|exists:categories,id',
            'sub_category_id' => 'nullable|integer|exists:sub_categories,id',
            'preview_text'    => 'required|string|max:255',
            'description'     => 'required|string',
            'director'        => 'required',
            'producer'        => 'required',
            'casts'           => 'required',
            'tags'            => 'required',
            'item_type'       => "$validation|in:1,2",
            'version'         => "nullable|required_if:item_type,==,1|in:$versions",
            'ratings'         => 'required|numeric',
            'rent_price'      => 'required_if:version,==,2|nullable|numeric|gte:0',
            'rental_period'   => 'required_if:version,==,2|nullable|integer|gte:0',
            'exclude_plan'    => 'required_if:version,==,2|nullable|in:0,1',
        ]);
    }

    private function imageUpload($request, $item, $type) {
        $landscape = @$item->image->landscape;
        $portrait  = @$item->image->portrait;

        if ($request->landscape_url) {
            $url      = $request->landscape_url;
            $contents = file_get_contents($url);
            $name     = substr($url, strrpos($url, '/') + 1);
            fileManager()->makeDirectory(getFilePath('item_landscape'));
            $path = getFilePath('item_landscape') . $name;
            Storage::put($name, $contents);
            File::move(storage_path('app/' . $name), $path);
            $landscape = $name;
        }

        if ($request->hasFile('landscape')) {
            $maxLandScapSize = $request->landscape->getSize() / 3000000;

            if ($maxLandScapSize > 3) {
                throw ValidationException::withMessages(['landscape' => 'Landscape image size could not be greater than 3mb']);
            }

            try {
                $date = date('Y') . '/' . date('m') . '/' . date('d');
                $type == 'update' ? fileManager()->removeFile(getFilePath('item_landscape') . @$item->image->landscape) : '';
                $landscape = $date . '/' . fileUploader($request->landscape, getFilePath('item_landscape') . $date);
            } catch (\Exception $e) {
                throw ValidationException::withMessages(['landscape' => 'Landscape image could not be uploaded']);
            }
        }

        if ($request->portrait_url) {
            $url      = $request->portrait_url;
            $contents = file_get_contents($url);
            $name     = substr($url, strrpos($url, '/') + 1);
            fileManager()->makeDirectory(getFilePath('item_portrait'));
            $path = getFilePath('item_portrait') . $name;
            Storage::put($name, $contents);
            File::move(storage_path('app/' . $name), $path);
            $portrait = $name;
        }

        if ($request->hasFile('portrait')) {
            $maxLandScapSize = $request->portrait->getSize() / 3000000;
            if ($maxLandScapSize > 3) {
                throw ValidationException::withMessages(['portrait' => 'Portrait image size could not be greater than 3mb']);
            }
            try {
                $date = date('Y') . '/' . date('m') . '/' . date('d');
                $type == 'update' ? fileManager()->removeFile(getFilePath('item_portrait') . @$item->image->portrait) : '';
                $portrait = $date . '/' . fileUploader($request->portrait, getFilePath('item_portrait') . $date);
            } catch (\Exception $e) {
                throw ValidationException::withMessages(['portrait' => 'Portrait image could not be uploaded']);
            }
        }
        $image = [
            'landscape' => $landscape,
            'portrait'  => $portrait,
        ];
        return $image;
    }

    protected function notifyToUsers($item) {
        $clickUrl  = route('watch', $item->slug);
        $users     = User::active()->cursor();
        $shortCode = [
            'title' => $item->title,
        ];
        foreach ($users as $user) {
            notify($user, 'SEND_ITEM_NOTIFICATION', $shortCode, redirectUrl: $clickUrl);
        }
    }

    protected function notifyToSubscribers($item) {
        $subscribers = Subscriber::orderBy('id', 'desc')->cursor();
        $clickUrl    = route('watch', $item->slug);
        $shortCode   = [
            'title' => $item->title,
        ];

        foreach ($subscribers as $subscriber) {
            $receiverName = explode('@', $subscriber->email)[0];
            $user         = [
                'username' => $subscriber->email,
                'email'    => $subscriber->email,
                'fullname' => $receiverName,
            ];
            notify($user, 'SEND_ITEM_NOTIFICATION_SUBSCRIBER', $shortCode, redirectUrl: $clickUrl);
        }
    }

    public function create() {
        $pageTitle  = "Add New Item";
        $categories = Category::active()->with([
            'subcategories' => function ($subcategory) {
                $subcategory->active();
            },
        ])->orderBy('id', 'desc')->get();
        return view('admin.item.create', compact('pageTitle', 'categories'));
    }

    public function store(Request $request) {
        $this->itemValidation($request, 'create');
        $team = [
            'director' => implode(',', $request->director),
            'producer' => implode(',', $request->producer),
            'casts'    => implode(',', $request->casts),
            'genres'   => implode(',', $request->genres),
            'language' => implode(',', $request->language),
        ];

        $item  = new Item();
        $image = $this->imageUpload($request, $item, 'update');

        $item->item_type = $request->item_type;
        $item->slug      = slug($request->title . '-' . now()->format('dmhis') . getTrx(5));
        $item->featured  = 0;

        $this->saveItem($request, $team, $image, $item);

        if ($request->notify_to_users) {
            $this->notifyToUsers($item);
        } else if ($request->notify_to_subscriber) {
            $this->notifyToSubscribers($item);
        }

        $notify[] = ['success', 'Item added successfully'];
        if ($request->item_type == Status::EPISODE_ITEM) {
            return redirect()->route('admin.item.episodes', $item->id)->withNotify($notify);
        } else {
            return redirect()->route('admin.item.uploadVideo', $item->id)->withNotify($notify);
        }
    }

    public function edit($id) {
        $item       = Item::findOrFail($id);
        $pageTitle  = "Edit : " . $item->title;
        $categories = Category::active()->with([
            'subcategories' => function ($subcategory) {
                $subcategory->active();
            },
        ])->orderBy('id', 'desc')->get();
        $subcategories = SubCategory::active()->where('category_id', $item->category_id)->orderBy('id', 'desc')->get();
        return view('admin.item.edit', compact('pageTitle', 'item', 'categories', 'subcategories'));
    }

    public function update(Request $request, $id) {
        $this->itemValidation($request, 'update');

        $team = [
            'director' => implode(',', $request->director),
            'producer' => implode(',', $request->producer),
            'casts'    => implode(',', $request->casts),
            'genres'   => implode(',', $request->genres),
            'language' => implode(',', $request->language),
        ];

        $item             = Item::findOrFail($id);
        $item->single     = $request->single ? Status::ENABLE : Status::DISABLE;
        $item->status     = $request->status ? Status::ENABLE : Status::DISABLE;
        $item->trending   = $request->trending ? Status::ENABLE : Status::DISABLE;
        $item->featured   = $request->featured ? Status::ENABLE : Status::DISABLE;
        $item->is_trailer = $request->is_trailer ? Status::ENABLE : Status::DISABLE;
        $image            = $this->imageUpload($request, $item, 'update');

        $this->saveItem($request, $team, $image, $item);

        $notify[] = ['success', 'Item updated successfully'];
        return back()->withNotify($notify);
    }

    private function saveItem($request, $team, $image, $item) {
        $item->category_id     = $request->category;
        $item->sub_category_id = $request->sub_category_id;
        $item->title           = $request->title;
        $item->preview_text    = $request->preview_text;
        $item->description     = $request->description;
        $item->team            = $team;
        $item->tags            = implode(',', $request->tags);
        $item->image           = $image;
        $item->version         = @$request->version ?? 0;
        $item->ratings         = $request->ratings;

        if ($request->version == Status::RENT_VERSION || $request->rent == Status::RENT_VERSION) {
            $item->rent_price    = @$request->rent_price ?? 0;
            $item->rental_period = @$request->rental_period ?? 0;
            $item->exclude_plan  = @$request->exclude_plan ?? 1;
        } else {
            $item->rent_price    = 0;
            $item->rental_period = 0;
            $item->exclude_plan  = 1;
        }

        $item->save();
    }

    public function uploadVideo($id) {
        $item  = Item::findOrFail($id);
        $video = $item->video;

        if ($video) {
            $notify[] = ['error', 'Already video exist'];
            return back()->withNotify($notify);
        }

        $pageTitle = "Upload video to: " . $item->title;
        $prevUrl   = route('admin.item.index');
        $route     = route('admin.item.upload.video', $item->id);
        return view('admin.item.video.upload', compact('item', 'pageTitle', 'video', 'prevUrl', 'route'));
    }

    public function upload(Request $request, $id) {
        $item = Item::where('id', $id)->first();
        if (!$item) {
            return response()->json(['error' => 'Item not found']);
        }

        $video = $item->video;

        if ($video) {
            $sevenTwentyLink  = 'nullable';
            $sevenTwentyVideo = 'nullable';
        } else {
            $sevenTwentyLink  = 'required_if:video_type_seven_twenty,0';
            $sevenTwentyVideo = 'required_if:video_type_seven_twenty,1';
        }

        ini_set('memory_limit', '-1');
        $validator = Validator::make($request->all(), [
            'video_type_three_sixty'     => 'required',
            'three_sixty_link'           => 'nullable',
            'three_sixty_video'          => ['nullable', new FileTypeValidate(['mp4', 'mkv', '3gp'])],

            'video_type_four_eighty'     => 'required',
            'four_eighty_link'           => 'nullable',
            'four_eighty_video'          => ['nullable', new FileTypeValidate(['mp4', 'mkv', '3gp'])],

            'video_type_seven_twenty'    => 'required',
            'seven_twenty_link'          => "$sevenTwentyLink",
            'seven_twenty_video'         => ["$sevenTwentyVideo", new FileTypeValidate(['mp4', 'mkv', '3gp'])],

            'video_type_thousand_eighty' => 'required',
            'thousand_eighty_link'       => 'nullable',
            'thousand_eighty_video'      => ['nullable', new FileTypeValidate(['mp4', 'mkv', '3gp'])],
        ], [
            'video_type_three_sixty'     => 'Video file 360P type is required',
            'three_sixty_link'           => 'Video file 360P link is required',
            'three_sixty_video'          => 'Video file 360P video is required',
            'video_type_four_eighty'     => 'Video file 480P type is required',
            'four_eighty_link'           => 'Video file 480P link is required',
            'four_eighty_video'          => 'Video file 480P video is required',
            'video_type_seven_twenty'    => 'Video file 720P type is required',
            'seven_twenty_link'          => 'Video file 720P link is required',
            'seven_twenty_video'         => 'Video file 720P video is required',
            'video_type_thousand_eighty' => 'Video file 1080P type is required',
            'thousand_eighty_link'       => 'Video file 1080P link is required',
            'thousand_eighty_video'      => 'Video file 1080P video is required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->all()]);
        }

        $sizeValidation = MultiVideoUploader::checkSizeValidation();
        if ($sizeValidation['error']) {
            return response()->json(['error' => $sizeValidation['message']]);
        }

        $uploadThreeSixty = MultiVideoUploader::multiQualityVideoUpload($video, 'three_sixty');
        if ($uploadThreeSixty['error']) {
            return response()->json(['error' => $sizeValidation['message']]);
        }

        $uploadFourEighty = MultiVideoUploader::multiQualityVideoUpload($video, 'four_eighty');
        if ($uploadFourEighty['error']) {
            return response()->json(['error' => $sizeValidation['message']]);
        }

        $uploadSevenTwenty = MultiVideoUploader::multiQualityVideoUpload($video, 'seven_twenty');
        if ($uploadSevenTwenty['error']) {
            return response()->json(['error' => $sizeValidation['message']]);
        }

        $uploadThousandEighty = MultiVideoUploader::multiQualityVideoUpload($video, 'thousand_eighty');
        if ($uploadThousandEighty['error']) {
            return response()->json(['error' => $sizeValidation['message']]);
        }

        if (!$video) {
            $video          = new Video();
            $video->item_id = $item->id;
        }

        $video->video_type_three_sixty     = @$request->video_type_three_sixty;
        $video->video_type_four_eighty     = @$request->video_type_four_eighty;
        $video->video_type_seven_twenty    = @$request->video_type_seven_twenty;
        $video->video_type_thousand_eighty = @$request->video_type_thousand_eighty;

        $video->three_sixty_video     = @$uploadThreeSixty['three_sixty_video'];
        $video->four_eighty_video     = @$uploadFourEighty['four_eighty_video'];
        $video->seven_twenty_video    = @$uploadSevenTwenty['seven_twenty_video'];
        $video->thousand_eighty_video = @$uploadThousandEighty['thousand_eighty_video'];

        $video->server_three_sixty     = @$uploadThreeSixty['server'] ?? 0;
        $video->server_four_eighty     = @$uploadFourEighty['server'] ?? 0;
        $video->server_seven_twenty    = @$uploadSevenTwenty['server'] ?? 0;
        $video->server_thousand_eighty = @$uploadThousandEighty['server'] ?? 0;

        $video->save();
        return response()->json(['success' => 'Video uploaded successfully']);
    }

    public function updateVideo(Request $request, $id) {
        $item  = Item::findOrFail($id);
        $video = $item->video;

        if (!$video) {
            $notify[] = ['error', 'Video not found'];
            return back()->withNotify($notify);
        }

        $pageTitle   = "Update video of: " . $item->title;
        $posterImage = getImage(getFilePath('item_landscape') . @$item->image->landscape);
        $general     = gs();

        $videoFile['three_sixty']     = getVideoFile($video, 'three_sixty');
        $videoFile['four_eighty']     = getVideoFile($video, 'four_eighty');
        $videoFile['seven_twenty']    = getVideoFile($video, 'seven_twenty');
        $videoFile['thousand_eighty'] = getVideoFile($video, 'thousand_eighty');

        $route   = route('admin.item.upload.video', @$item->id);
        $prevUrl = route('admin.item.index');
        return view('admin.item.video.update', compact('item', 'pageTitle', 'video', 'videoFile', 'posterImage', 'prevUrl', 'route'));
    }

    public function itemList(Request $request) {
        $items = Item::hasVideo();

        if (request()->search) {
            $items = $items->where('title', 'like', "%$request->search%");
        }

        $items = $items->latest()->paginate(getPaginate());

        foreach ($items as $item) {
            $response[] = [
                'id'   => $item->id,
                'text' => $item->title,
            ];
        }

        return $response ?? [];
    }

    public function itemFetch(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'        => 'required|integer',
            'item_type' => 'required|integer|in:1,2',
        ]);
        if ($validate->fails()) {
            return response()->json(['error' => $validate->errors()]);
        }
        $general  = gs();
        $itemType = $request->item_type == 1 ? 'movie' : 'tv';
        $tmDbUrl  = 'https://api.themoviedb.org/3/' . $itemType . '/' . $request->id;
        $url      = $tmDbUrl . '?api_key=' . $general->tmdb_api;
        $castUrl  = $tmDbUrl . '/credits?api_key=' . $general->tmdb_api;
        $tags     = $tmDbUrl . '/keywords?api_key=' . $general->tmdb_api;

        $movieResponse = CurlRequest::curlContent($url);
        $castResponse  = CurlRequest::curlContent($castUrl);
        $tagsResponse  = CurlRequest::curlContent($tags);

        $data  = json_decode($movieResponse);
        $casts = json_decode($castResponse);
        $tags  = json_decode($tagsResponse);

        if (isset($data->success)) {
            return response()->json(['error' => 'The resource you requested could not be found.']);
        }

        return response()->json([
            'success' => true,
            'data'    => $data,
            'casts'   => $casts,
            'tags'    => $tags,
        ]);
    }

    public function sendNotification($id) {
        $item = Item::where('status', Status::ENABLE)->findOrFail($id);
        $this->notifyToUsers($item);
        $notify[] = ['success', 'Notification send successfully'];
        return back()->withNotify($notify);
    }

    public function adsDuration($id, $episodeId = 0) {
        $pageTitle = 'Ads Configuration';
        $item      = Item::findOrFail($id);

        if ($item->item_type == 1 || $item->item_type == 3) {
            $episodeId = null;
            $video     = $item->video;
        } else {
            $episode   = $item->episodes()->where('id', $episodeId)->with('video')->first();
            $video     = $episode->video;
            $episodeId = $episode->id;
        }
        $general   = gs();
        $videoFile = getVideoFile($video, 'seven_twenty');
        return view('admin.item.video.ads', compact('pageTitle', 'item', 'video', 'videoFile', 'episodeId'));
    }

    public function adsDurationStore(Request $request, $id = 0, $episodeId = 0) {
        $request->validate([
            'ads_time'   => 'required|array',
            'ads_time.*' => 'required',
        ]);
        $item = Item::findOrFail($id);
        if ($item->item_type == 1 || $item->item_type == 3) {
            $video = $item->video;
        } else {
            $episode = $item->episodes()->with('video')->findOrFail($episodeId);
            $video   = $episode->video;
        }
        for ($i = 0; $i < count($request->ads_time); $i++) {
            $arr = explode(':', $request->ads_time[$i]);
            if (count($arr) === 3) {
                $second[] = $arr[0] * 3600 + $arr[1] * 60 + $arr[2];
            } else {
                $second[] = $arr[0] * 60 + $arr[1];
            }
        }
        $video->seconds  = $second;
        $video->ads_time = $request->ads_time;
        $video->save();

        $notify[] = ['success', 'Ads time added successfully'];
        return back()->withNotify($notify);
    }

    public function subtitles($id, $videoId = 0) {
        $itemId    = 0;
        $episodeId = 0;
        if ($videoId == 0) {
            $item       = Item::with('video')->findOrFail($id);
            $videoId    = $item->video->id;
            $columnName = 'item_id';
            $itemId     = $item->id;
        } else {
            $item       = Episode::with('video')->findOrFail($id);
            $columnName = 'episode_id';
            $episodeId  = $item->id;
        }
        $subtitles = Subtitle::where($columnName, $id)->where('video_id', $videoId)->paginate(getPaginate());
        $pageTitle = 'Subtitles for - ' . $item->title;
        return view('admin.item.video.subtitles', compact('pageTitle', 'item', 'subtitles', 'videoId', 'episodeId', 'itemId'));
    }

    public function subtitleStore(Request $request, $itemId, $episodeId, $videoId, $id = 0) {
        $validate = $id ? 'nullable' : 'required';
        $request->validate([
            'language' => 'required|string|max:40',
            'code'     => 'required|string|max:40',
            'file'     => [$validate, new FileTypeValidate(['vtt'])],
        ]);

        if ($id) {
            $subtitle     = Subtitle::findOrFail($id);
            $oldFile      = $subtitle->file;
            $notification = 'Subtitle updated successfully';
        } else {
            $subtitle     = new Subtitle();
            $notification = 'Subtitle created successfully';
            $oldFile      = null;
        }

        $subtitle->item_id    = $itemId;
        $subtitle->episode_id = $episodeId;
        $subtitle->video_id   = $videoId;
        $subtitle->language   = $request->language;
        $subtitle->code       = strtolower($request->code);
        if ($request->file) {
            $subtitle->file = fileUploader($request->file, getFilePath('subtitle'), null, $oldFile);
        }
        $subtitle->save();
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function subtitleDelete($id) {
        $subtitle = Subtitle::where('id', $id)->firstOrFail();
        fileManager()->removeFile(getFilePath('subtitle') . '/' . $subtitle->file);
        $subtitle->delete();
        $notify[] = ['success', 'Subtitle deleted successfully'];
        return back()->withNotify($notify);
    }

    public function report($id, $videoId = 0) {
        $startDate = now()->subMonth()->startOfDay()->toDateString();
        $endDate   = now()->endOfDay()->toDateString();

        if ($videoId == 0) {
            $item         = Item::with('videoReport')->findOrFail($id);
            $videoReports = $item->videoReport()->whereBetween('created_at', [$startDate, $endDate])->get();
            $title        = $item->title;
        } else {
            $episode      = Episode::with('item', 'videoReport')->findOrFail($videoId);
            $item         = $episode->item;
            $videoReports = $episode->videoReport()->whereBetween('created_at', [$startDate, $endDate])->get();
            $title        = $episode->title;
        }

        $reports = $videoReports->groupBy(function ($data) {
            return substr($data['created_at'], 0, 10);
        })->map(function ($group) {
            return count($group);
        });

        $totalViews = $videoReports->count();

        $allRegions = VideoReport::distinct('region_name')->pluck('region_name');
        $pageTitle  = 'Report - ' . $item->title;

        return view('admin.item.report', compact('pageTitle', 'reports', 'item', 'totalViews', 'title', 'allRegions', 'videoId'));
    }

    public function fetchViewReportData(Request $request) {
        $startDate = Carbon::parse($request->start_date)->startOfDay();
        $endDate   = Carbon::parse($request->end_date)->endOfDay();

        $dates = $this->getAllDates($startDate, $endDate);

        $mainQuery = VideoReport::where('created_at', '>=', $startDate)->where('created_at', '<=', $endDate);
        if ($request->episode_id == 0) {
            $videoReports = $mainQuery->where('item_id', $request->item_id);
            if ($request->search) {
                $videoReports = $videoReports->where('region_name', $request->search);
            }
        } else {
            $videoReports = $mainQuery->where('episode_id', $request->episode_id);
            if ($request->search) {
                $videoReports = $videoReports->where('region_name', $request->search);
            }
        }

        $videoReports = $videoReports->get();
        $totalViews   = $videoReports->count();

        $groupedReports = $videoReports->groupBy(function ($data) {
            return $data->created_at->format('Y-m-d');
        })->map(function ($group) {
            return $group->count();
        });

        $data = [];
        foreach ($dates as $date) {
            $data[] = [
                'created_on' => $date,
                'views'      => $groupedReports[$date] ?? 0,
            ];
        }

        return response()->json([
            'created_on' => collect($data)->pluck('created_on'),
            'data'       => [
                [
                    'name' => 'Total Views',
                    'data' => collect($data)->pluck('views'),
                ],
                'fetch_total_view' => $totalViews,
            ],
        ]);
    }

    private function getAllDates($startDate, $endDate) {
        $dates       = [];
        $currentDate = new \DateTime($startDate);
        $endDate     = new \DateTime($endDate);

        while ($currentDate <= $endDate) {
            $dates[] = $currentDate->format('Y-m-d');
            $currentDate->modify('+1 day');
        }

        return $dates;
    }

    public function import(Request $request) {
        $request->validate([
            'file' => 'required|file|mimes:csv',
        ]);
        $file = $request->file('file');
        if (($handle = fopen($file, 'r')) !== false) {
            $headings = fgetcsv($handle);

            if (isset($headings[0])) {
                $headings[0] = preg_replace('/^\x{FEFF}/u', '', $headings[0]);
            }

            $versions = implode(',', [Status::FREE_VERSION, Status::PAID_VERSION, Status::RENT_VERSION]);
            $items    = [];
            $errors   = [];

            while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                $row       = array_combine($headings, $data);
                $validator = Validator::make($row, [
                    'title'           => 'required|string|max:255',
                    'category_id'     => 'required|integer|exists:categories,id',
                    'sub_category_id' => 'nullable|integer|exists:sub_categories,id',
                    'preview_text'    => 'required|string|max:255',
                    'description'     => 'required|string',
                    'item_type'       => "required|in:1,2",
                    'view'            => 'nullable|integer',
                    'version'         => "nullable|required_if:item_type,==,1|in:$versions",
                    'ratings'         => 'required|numeric',
                    'rent_price'      => 'required_if:version,==,2|nullable|numeric|gte:0',
                    'rental_period'   => 'required_if:version,==,2|nullable|integer|gte:0',
                    'exclude_plan'    => 'required_if:version,==,2|nullable|in:0,1',
                ]);

                if ($validator->fails()) {
                    $errors[] = $validator->errors()->all();
                    continue;
                }

                $items[] = [
                    'category_id'     => @$row['category_id'],
                    'sub_category_id' => @$row['sub_category_id'],
                    'slug'            => @$row['slug'],
                    'title'           => @$row['title'],
                    'preview_text'    => @$row['preview_text'],
                    'description'     => @$row['description'],
                    'team'            => @$row['team'],
                    'image'           => @$row['image'],
                    'item_type'       => @$row['item_type'],
                    'status'          => @$row['status'],
                    'single'          => @$row['single'],
                    'trending'        => @$row['trending'],
                    'featured'        => @$row['featured'],
                    'version'         => @$row['version'],
                    'tags'            => @$row['tags'],
                    'ratings'         => @$row['ratings'],
                    'view'            => @$row['view'],
                    'is_trailer'      => @$row['is_trailer'],
                    'rent_price'      => @$row['rent_price'],
                    'rental_period'   => @$row['rental_period'],
                    'exclude_plan'    => @$row['exclude_plan'],
                    'created_at'      => @$row['created_at'],
                    'updated_at'      => @$row['updated_at'],
                ];
            }
            fclose($handle);
        }
        if (!blank($errors)) {
            foreach ($errors as $error) {
                $notify[] = ['error', implode(', ', $error)];
            }
            return back()->withNotify($notify);
        }
        if (!blank($items)) {
            Item::insert($items);
        }

        $notify[] = ['success', 'CSV data imported successfully!'];
        return back()->withNotify($notify);
    }

    public function reels() {
        $pageTitle = 'Reels Item';
        $reels     = Reel::searchable(['title'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.item.reels', compact('pageTitle', 'reels'));
    }

    public function reelStore(Request $request, $id = 0) {
        $videoValidate = $id ? 'nullable' : 'required';
        $request->validate([
            'title'       => 'required|string|max:255',
            'description' => 'required|string|max:255',
            'video'       => [$videoValidate, new FileTypeValidate(['mp4'])],
        ]);

        if ($id) {
            $reel         = Reel::findOrFail($id);
            $notification = 'Reel updated successfully';
        } else {
            $reel         = new Reel();
            $notification = 'Reel created successfully';
        }
        $reel->title       = $request->title;
        $reel->description = $request->description;
        if ($request->hasFile('video')) {
            try {
                $reel->video = fileUploader($request->video, getFilePath('reels'), null, @$reel->video);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }
        $reel->save();
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function reelRemove($id) {
        $reel = Reel::findOrFail($id);
        fileManager()->removeFile(getFilePath('reels') . '/' . @$reel->video);
        $reel->delete();
        $notify[] = ['success', 'Reel deleted successfully'];
        return back()->withNotify($notify);
    }

    public function seo($id) {
        $key       = 'item';
        $data      = Item::findOrFail($id);
        $pageTitle = 'SEO Configuration: ' . $data->title;
        return view('admin.item.seo', compact('pageTitle', 'key', 'data'));
    }

    public function seoStore(Request $request, $id) {
        $request->validate([
            'image' => ['nullable', new FileTypeValidate(['jpeg', 'jpg', 'png'])],
        ]);

        $data  = Item::findOrFail($id);
        $image = @$data->seo_content->image;
        if ($request->hasFile('image')) {
            try {
                $path  = getFilePath('itemSeo');
                $image = fileUploader($request->image, $path, getFileSize('seo'), @$data->seo_content->image);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload the seo image'];
                return back()->withNotify($notify);
            }
        }
        $data->seo_content = [
            'image'              => $image,
            'description'        => $request->description,
            'social_title'       => $request->social_title,
            'social_description' => $request->social_description,
            'keywords'           => $request->keywords,
        ];
        $data->save();

        $notify[] = ['success', 'SEO content has been updated successfully'];
        return to_route('admin.item.seo', $data->id)->withNotify($notify);
    }
}
