import __ from "@/Functions/Translate";
import PrimaryButton from "@/Components/PrimaryButton";
import SecondaryButton from "@/Components/SecondaryButton";
import DangerButton from "@/Components/DangerButton";
import { Head, Link } from "@inertiajs/inertia-react";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import { Inertia } from "@inertiajs/inertia";
import { AiOutlineEdit } from "react-icons/ai";
import { RiDeleteBin5Line } from "react-icons/ri";
import { Tooltip } from "react-tooltip";
import "react-tooltip/dist/react-tooltip.css";
import { useState } from "react";
import Modal from "@/Components/Modal";
import { BsTagFill } from "react-icons/bs";
import AccountNavi from "../Channel/Partials/AccountNavi";
import { MdOutlineVideoLibrary } from "react-icons/md";
import { FaPhotoVideo } from "react-icons/fa";

export default function GalleryManager({ galleries }) {
  const [showDeleteConfirmation, setShowDeleteConfirmation] = useState(false);
  const [deleteId, setDeleteId] = useState(0);
  const [isHovered, setIsHovered] = useState(null);

  const confirmDelete = (e, id) => {
    e.preventDefault();

    setShowDeleteConfirmation(true);
    setDeleteId(id);
  };

  const deleteGallery = () => {
    Inertia.visit(route("gallery.delete"), {
      method: "POST",
      data: { gallery: deleteId },
      preserveState: false,
    });
  };

  // active tab class
  const activeTabClass =
    "text-xl font-bold mr-2 md:mr-4 text-indigo-800 dark:text-indigo-500 border-b-2 border-indigo-800";
  const inactiveTabClass =
    "text-xl font-bold mr-2 md:mr-4 hover:text-indigo-800 dark:text-white dark:hover:text-indigo-500";

  return (
    <AuthenticatedLayout>
      <Head title={__("Galleries")} />

      <Modal
        show={showDeleteConfirmation}
        onClose={(e) => setShowDeleteConfirmation(false)}
      >
        <div className="px-5 py-10 text-center">
          <h3 className="text-xl mb-3 text-zinc-700 dark:text-white">
            {__("Are you sure you want to remove this Gallery?")}
          </h3>
          <DangerButton onClick={(e) => deleteGallery()}>
            {__("Yes")}
          </DangerButton>
          <SecondaryButton
            className="ml-3"
            onClick={(e) => setShowDeleteConfirmation(false)}
          >
            {__("No")}
          </SecondaryButton>
        </div>
      </Modal>

      <div className="lg:flex lg:space-x-10">
        <AccountNavi active={"gallery-manager"} />
        <div className="ml-0 w-full ">
          <Link href={route("gallery.manager")} className={activeTabClass}>
            {__("Gallery Manager")}
          </Link>
          <Link href={route("gallery.ordered")} className={inactiveTabClass}>
            {__("Galleries Ordered")}
          </Link>

          <div className="mt-5 p-4 sm:p-8 bg-white dark:bg-zinc-900 shadow sm:rounded-lg">
            <header>
              <h2 className="text-lg inline-flex items-center md:text-xl font-medium text-gray-600 dark:text-gray-100">
                <MdOutlineVideoLibrary className="mr-2" />
                {__("My Galleries")}
              </h2>

              <p className="mt-1 mb-2 text-sm text-gray-600 dark:text-gray-400">
                {__("Upload & manage galleries for the channel")}
              </p>

              <PrimaryButton
                onClick={(e) => Inertia.visit(route("gallery.create"))}
              >
                {__("Create Gallery")}
              </PrimaryButton>
            </header>

            <hr className="my-5" />

            {galleries.total === 0 && (
              <div className="text-gray-600 dark:text-white">
                {__("You didn't create any galleries yet.")}
              </div>
            )}

            {galleries.total !== 0 && (
              <div className="flex md:space-x-5 flex-wrap">
                {galleries.data.map((v) => (
                  <div
                    key={`gallery-${v.id}`}
                    className={"mt-5"}
                    onMouseEnter={() => setIsHovered(v.id)}
                    onMouseLeave={() => setIsHovered(null)}
                  >
                    <div className="relative">
                      <img
                        src={v.previewPhotoUrl}
                        alt=""
                        className="rounded-lg"
                      />

                      {isHovered == v.id && (
                        <div className="absolute inset-0 flex items-center justify-center bg-black bg-opacity-50 transition-opacity opacity-100 group-hover:opacity-100">
                          <Link
                            href={route("gallery.edit", {
                              gallery: v.id,
                            })}
                            className="bg-indigo-600 flex items-center text-white px-4 py-2 rounded-lg"
                          >
                            <FaPhotoVideo className="w-5 h-5 mr-2 text-white" />
                            {__("Manage Photos")}
                          </Link>
                        </div>
                      )}
                    </div>

                    <div className="my-3 h-6 overflow-hidden text-gray-600 text-sm font-semibold dark:text-white">
                      <a
                        data-tooltip-content={v.gallery_name}
                        data-tooltip-id={`tooltip-${v.id}`}
                      >
                        {v.gallery_name}
                      </a>

                      <Tooltip anchorSelect="a" />
                    </div>

                    <div className="mt-3 flex items-center space-x-2 text-sm justify-between">
                      <span className="text-gray-600 dark:text-white">
                        {__("Price")}{" "}
                      </span>
                      <span className="px-2 py-1 text-sm rounded-lg bg-sky-500 text-white">
                        {v.price > 0
                          ? __(":tokensPrice tokens", {
                              tokensPrice: v.price,
                            })
                          : __("Free")}
                      </span>
                    </div>

                    <div className="mt-2 flex items-center space-x-2 text-sm justify-between">
                      <span className="text-gray-600 dark:text-white">
                        {__("Photos")}{" "}
                      </span>
                      <span className="px-2 py-1 rounded-lg bg-amber-500 text-white">
                        {v.photos_count}
                      </span>
                    </div>

                    <div className="mt-2 flex items-center space-x-2 text-sm justify-between">
                      <span className="text-gray-600 dark:text-white">
                        {__("Free for subs")}{" "}
                      </span>
                      <span className="px-2 py-1 rounded-lg bg-teal-500 text-white">
                        {v.free_for_subs}
                      </span>
                    </div>

                    <div className="flex mt-2 items-center space-x-2 text-sm justify-between">
                      <span className="text-gray-600 dark:text-white">
                        {__("Earnings")}{" "}
                      </span>
                      <span className="px-2 py-1 rounded-lg bg-pink-500 text-white">
                        {v.sales_sum_price
                          ? __(":salesTokens tokens", {
                              salesTokens: v.sales_sum_price,
                            })
                          : "--"}
                      </span>
                    </div>

                    <div className="border-t pt-3 mt-3  flex items-center">
                      <Link
                        href={route("gallery.edit", {
                          gallery: v.id,
                        })}
                      >
                        <AiOutlineEdit className="w-6 h-6 mr-2 text-sky-600" />
                      </Link>

                      <Link
                        href={route("gallery.edit", {
                          gallery: v.id,
                        })}
                      >
                        <FaPhotoVideo className="w-6 h-6 mr-2 text-amber-600" />
                      </Link>

                      <button onClick={(e) => confirmDelete(e, v.id)}>
                        <RiDeleteBin5Line className="text-red-600 w-5 h-5" />
                      </button>
                    </div>
                  </div>
                ))}
              </div>
            )}

            {galleries.last_page > 1 && (
              <>
                <hr className="my-5" />

                <div className="flex text-gray-600 my-3 text-sm">
                  {__("Page: :pageNumber of :lastPage", {
                    pageNumber: galleries.current_page,
                    lastPage: galleries.last_page,
                  })}
                </div>

                <SecondaryButton
                  processing={galleries.prev_page_url ? false : true}
                  className="mr-3"
                  onClick={(e) => Inertia.visit(galleries.prev_page_url)}
                >
                  {__("Previous")}
                </SecondaryButton>

                <SecondaryButton
                  processing={galleries.next_page_url ? false : true}
                  onClick={(e) => Inertia.visit(galleries.next_page_url)}
                >
                  {__("Next")}
                </SecondaryButton>
              </>
            )}
          </div>
        </div>
      </div>
    </AuthenticatedLayout>
  );
}
