<?php

namespace App\Http\Controllers;

use App\Models\Galleries;
use App\Models\GalleryPhotos;
use App\Models\GallerySales;
use App\Models\User;
use App\Notifications\AffiliateEarningNotification;
use App\Notifications\NewGallerySale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Inertia\Inertia;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class GalleryController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth')->except(['browseGalleries', 'userGalleries', 'galleryPhotos']);
    }

    public function browseGalleries()
    {
        $exploreImage = asset('images/browse-galleries-icon.png');
        return Inertia::render('Galleries/Browse', [
            'galleries' => Galleries::with('user')
                ->withCount(['photos'])
                ->latest()
                ->paginate(9),
            'exploreImage' => $exploreImage
        ]);
    }

    public function unlockGallery(Galleries $gallery, Request $r)
    {
        $gallery->load('user');

        if ($gallery->canBeViewed) {
            return back()->with('message', __('You already have access to this gallery'));
        }

        return Inertia::render('Galleries/GalleryUnlock', compact('gallery'));
    }

    public function purchaseGallery(Galleries $gallery, Request $request)
    {
        // check if user already bought
        if ($gallery->canBeViewed) {
            return back()->with('message', __('You already have access to this gallery'));
        }

        // enough balance?
        if ($gallery->price > $request->user()->tokens) {
            return redirect(route('token.packages'))->with('message', __("Your tokens balance does not cover :amount tokens required to unlock the gallery", ['amount' => $gallery->price]));
        }

        // tokens
        $tokens = $gallery->price;
        $user = $gallery->user;

        // record order
        $videoSale = new GallerySales();
        $videoSale->gallery_id = $gallery->id;
        $videoSale->streamer_id = $gallery->user_id;
        $videoSale->user_id = $request->user()->id;
        $videoSale->price = $tokens;
        $videoSale->save();

        // decrement buyer balance
        $request->user()->decrement('tokens', $gallery->price);

        // increment streamer balance
        $gallery->user->increment('tokens', $tokens);

        // notify streamer of this sale (on platform)
        $gallery->user->notify(new NewGallerySale($videoSale));

        // redirect to my videos
        return redirect(route('gallery.ordered'))->with('message', __("Thank you, you can now access the gallery!"));
    }

    // galleries for user
    public function userGalleries(User $user)
    {
        return $user->galleries()
            ->with('user')
            ->withCount(['photos'])
            ->paginate(9);
    }


    // get photos in a gallery
    public function galleryPhotos(Galleries $gallery, Request $r)
    {

        if (!$gallery->canBeViewed) {
            abort(403);
        }

        return $gallery->photos;
    }

    // user ordered galleries
    public function myGalleries(Request $request)
    {

        $galleries = $request->user()
            ->purchasedGalleries()
            ->withCount(['photos'])
            ->with(['user', 'photos']);

        if ($request->has('search_term')) {
            $galleries->where('gallery_name', 'LIKE', '%' . $request->search_term . '%');
        }

        $galleries = $galleries->paginate(4);

        return Inertia::render('Galleries/GalleriesOrdered', compact('galleries'));
    }

    // streamer gallery manager
    public function galleryManager(Request $request)
    {

        Gate::authorize('channel-settings');

        $galleries = $request->user()->galleries()
            ->withSum('sales', 'price')
            ->withCount(['photos'])
            ->orderByDesc('id')
            ->paginate(9);


        return Inertia::render('Galleries/GalleryManager', compact('galleries'));
    }

    // create gallery
    public function createGallery(Request $request)
    {

        Gate::authorize('channel-settings');

        $gallery = [
            'id' => null,
            'gallery_name' => '',
            'price' => 0,
            'preview_photo' => '',
            'free_for_subs' => 'no'
        ];

        return Inertia::render('Galleries/Create', compact('gallery'));
    }

    // save gallery
    public function saveGallery(Request $request)
    {
        Gate::authorize('channel-settings');

        $request->validate([
            'gallery_name' => 'required|min:2',
            'price' => 'required|numeric',
            'free_for_subs' => 'required|in:yes,no',
            'preview_photo' => 'required|mimes:png,jpg',
        ]);

        // resize & upload preview image
        $thumbnail = Image::make($request->file('preview_photo'))->fit(220, 340)->stream();
        $thumbFile = 'thumbnails/' . uniqid() . '-' . auth()->id() . '.' . $request->file('preview_photo')->getClientOriginalExtension();

        Storage::disk(env('FILESYSTEM_DISK'))->put($thumbFile, $thumbnail);
        Storage::disk(env('FILESYSTEM_DISK'))->setVisibility($thumbFile, 'public');

        // create gallery entry
        $request->user()->galleries()->create([
            'gallery_name' => $request->gallery_name,
            'price' => $request->price,
            'free_for_subs' => $request->free_for_subs,
            'preview_photo' => $thumbFile,
            'disk' => env('FILESYSTEM_DISK'),
        ]);

        return to_route('gallery.manager')->with('message', __('Gallery successfully created. You may now add photos to it.'));
    }

    // create gallery
    public function editGallery(Galleries $gallery, Request $request)
    {

        $gallery->load('photos');

        Gate::authorize('channel-settings');



        return Inertia::render('Galleries/Create', compact('gallery'));
    }


    public function updateGallery(Galleries $gallery, Request $request)
    {
        Gate::authorize('channel-settings');

        $request->validate([
            'gallery_name' => 'required|min:2',
            'price' => 'required|numeric',
            'free_for_subs' => 'required|in:yes,no',
        ]);

        if ($request->user()->id !== $gallery->user_id) {
            abort(403, __("You do not seem to be the owner of this gallery"));
        }

        // resize & upload thumbnail
        if ($request->hasFile('preview_photo')) {

            $request->validate([
                'preview_photo' => 'required|mimes:png,jpg',
            ]);

            $thumbnail = Image::make($request->file('preview_photo'))->fit(220, 340)->stream();
            $thumbFile = 'thumbnails/' . uniqid() . '-' . auth()->id() . '.' . $request->file('preview_photo')->getClientOriginalExtension();

            Storage::disk(env('FILESYSTEM_DISK'))->put($thumbFile, $thumbnail);
            Storage::disk(env('FILESYSTEM_DISK'))->setVisibility($thumbFile, 'public');

            $gallery->preview_photo = $thumbFile;
            $gallery->disk = env('FILESYSTEM_DISK');
            $gallery->save();
        }

        // update the gallery entry
        $gallery->update([
            'gallery_name' => $request->gallery_name,
            'price' => $request->price,
            'free_for_subs' => $request->free_for_subs,
        ]);

        return back()->with('message', __('Gallery successfully updated'));
    }

    // upload photos
    public function uploadPhotos(Galleries $gallery, Request $request)
    {

        Gate::authorize('channel-settings');

        $request->validate([
            'photos' => 'required|array|min:1', // 'photos' should be an array with at least one element
            'photos.*' => 'image|mimes:jpeg,png,jpg,gif,svg|max:2048', // Each element in 'photos' should be an image file
        ]);


        // add photos to the gallery
        foreach ($request->file('photos') as $uploadedPhoto) {

            $uniqueId = uniqid();

            // upload original first
            $originalFileName = $uniqueId . '-' . auth()->id() . "." .  $uploadedPhoto->getClientOriginalExtension();
            $originalPath = $uploadedPhoto->storeAs('gallery-photos', $originalFileName, 'public');

            $thumbnail = Image::make($uploadedPhoto)->fit(220, 340)->stream();
            $thumbFile = 'gallery-photos/thumbnails/' . $uniqueId . '-' . auth()->id() . '.' . $uploadedPhoto->getClientOriginalExtension();

            Storage::disk(env('FILESYSTEM_DISK'))->put($thumbFile, $thumbnail);
            Storage::disk(env('FILESYSTEM_DISK'))->setVisibility($thumbFile, 'public');

            // Save the relationship
            $gallery->photos()->create([
                'user_id' => auth()->id(),
                'photo' => $originalPath,
                'disk' => env('FILESYSTEM_DISK')
            ]);
        }
    }

    // delete a single photo from the gallery
    public function deletePhotoFromGallery(Request $request)
    {

        Gate::authorize('channel-settings');

        // find photo
        $photo = $request->user()->photos()->findOrFail($request->photo);

        // delete file from disk
        Storage::disk($photo->disk)->delete($photo->photo);

        // delete photo
        $photo->delete();

        return back()->with('message', __('Photo removed from the gallery'));
    }

    // delete entire gallery
    public function deleteGallery(Galleries $gallery, Request $request)
    {

        Gate::authorize('channel-settings');

        // find gallery
        $gallery = $request->user()->galleries()->findOrFail($request->gallery);

        // delete file from disk
        Storage::disk($gallery->disk)->delete($gallery->preview_photo);

        // delete gallery sales
        $gallery->sales()->delete();

        // delete gallery photos
        foreach ($gallery->photos as $p) {
            Storage::disk($p->disk)->delete($p->photo);
            $p->delete();
        }

        // delete gallery
        $gallery->delete();

        return back()->with('message', __('Gallery removed'));
    }
}
